package kiyut.sketsa.modules.symbollibrary.general;

import java.awt.Component;
import java.awt.Point;
import java.awt.geom.AffineTransform;
import java.awt.geom.Area;
import java.awt.geom.Ellipse2D;
import java.awt.geom.NoninvertibleTransformException;
import java.awt.geom.PathIterator;
import java.awt.geom.Rectangle2D;
import java.util.ArrayList;
import java.util.List;
import javax.swing.Icon;
import javax.swing.ImageIcon;
import javax.swing.JOptionPane;
import kiyut.sketsa.canvas.VectorCanvas;
import kiyut.sketsa.undo.DOMUndoManager;
import kiyut.sketsa.util.ColorConvertion;
import kiyut.sketsa.util.DOMUtilities;
import kiyut.sketsa.util.SVGConstants;
import org.w3c.dom.Element;
import org.w3c.dom.svg.SVGDocument;
import org.w3c.dom.svg.SVGElement;
import org.w3c.dom.svg.SVGPathElement;
import org.w3c.dom.svg.SVGPathSegClosePath;
import org.w3c.dom.svg.SVGPathSegCurvetoCubicAbs;
import org.w3c.dom.svg.SVGPathSegLinetoAbs;
import org.w3c.dom.svg.SVGPathSegMovetoAbs;


/** Implementation of Heart Symbol
 *
 * @author Kiyut
 */
public class HeartSymbol extends GeneralSymbol {
    /** Icon */
    protected Icon icon;
    
    /** Tooltip Text */
    protected String tooltipText;
    
    protected double radius;
    
    /** Creates a new instance of HeartSymbol */
    public HeartSymbol() {
        icon = new ImageIcon(getClass().getResource("/kiyut/sketsa/modules/symbollibrary/general/heart32.png"));
        tooltipText = "Heart";
        radius = 50;
    }
    
    @Override
    public Icon getIcon() {
        return icon;
    }
    
    @Override
    public String getTooltipText() {
        return tooltipText;
    }
    
    @Override
    public void insertSymbol(VectorCanvas canvas, Point location) {
        SVGElement elt = createSVGElement(canvas,location);
        SVGDocument doc = canvas.getSVGDocument();
        Element svgRoot = doc.getDocumentElement();
        
        DOMUndoManager undoManager = canvas.getUndoManager();
        undoManager.start("Add " + tooltipText);
        try {
            svgRoot.appendChild(elt);
        } finally {
            undoManager.end();
        }
        
        List<SVGElement> newList = new ArrayList<>();
        newList.add(elt);
        
        canvas.getCanvasSelection().setSelectionList(newList);
        canvas.refresh();
    }
    
    /** Create SVGElement
     * @param canvas VectorCanvas
     * @param location The location where the insertion should take place
     * @return SVGElement
     */
    protected SVGElement createSVGElement(VectorCanvas canvas, Point location) {
        SVGDocument doc = canvas.getSVGDocument();
        
        float[] coords = new float[6];
        coords[0] = (float)location.getX();
        coords[1] = (float)location.getY();
        
        AffineTransform at = canvas.getTransform(true);
        try {
            at = at.createInverse();
        } catch (NoninvertibleTransformException e) {}
        at.transform(coords,0,coords,0,2);
        
        float cx = coords[0];
        float cy = coords[1];
        
        PathIterator pi = generateShape(cx,cy).getPathIterator(null);
        
        SVGPathElement elt = (SVGPathElement)doc.createElementNS(SVGConstants.SVG_NAMESPACE_URI, SVGConstants.SVG_PATH_TAG);
        
        // adding the path seg
        while (!pi.isDone()) {
            int segment = pi.currentSegment(coords);
            if (segment == PathIterator.SEG_MOVETO) {
                SVGPathSegMovetoAbs svgPathSeg = elt.createSVGPathSegMovetoAbs(coords[0],coords[1]);
                elt.getPathSegList().appendItem(svgPathSeg);
            } else if (segment == PathIterator.SEG_CLOSE) {
                SVGPathSegClosePath svgPathSeg = elt.createSVGPathSegClosePath();
                elt.getPathSegList().appendItem(svgPathSeg);
            } else if (segment == PathIterator.SEG_LINETO) {
                SVGPathSegLinetoAbs svgPathSeg = elt.createSVGPathSegLinetoAbs(coords[0],coords[1]);
                elt.getPathSegList().appendItem(svgPathSeg);
            } else if (segment == PathIterator.SEG_CUBICTO) {
                SVGPathSegCurvetoCubicAbs svgPathSeg = elt.createSVGPathSegCurvetoCubicAbs(coords[4],coords[5],coords[0],coords[1],coords[2],coords[3]);
                elt.getPathSegList().appendItem(svgPathSeg);
            }
            
            pi.next();
        }
        
        DOMUtilities.updateProperty(elt, null, SVGConstants.SVG_STROKE_ATTRIBUTE,SVGConstants.SVG_NONE_VALUE);
        DOMUtilities.updateProperty(elt, null, SVGConstants.SVG_FILL_ATTRIBUTE,ColorConvertion.toHexString(java.awt.Color.RED));
        
        return elt;
    }
    
    /** It is created by combining rotated square with 2 circle
     * @param cx center x coord
     * @param cy center y coord
     * @return Area
     */
    protected Area generateShape(double cx,double cy) {
        Rectangle2D r = new Rectangle2D.Double(cx,cy,radius,radius);
        Ellipse2D ell1 = new Ellipse2D.Double(cx-(radius/2),cy,radius,radius);
        Ellipse2D ell2 = new Ellipse2D.Double(cx,cy-(radius/2),radius,radius);
        
        Area area = new Area(r);
        area.add(new Area(ell1));
        area.add(new Area(ell2));
        area.transform(AffineTransform.getRotateInstance(Math.toRadians(45),cx,cy));
        
        // center the area
        r = area.getBounds2D();
        double tx = cx - r.getCenterX();
        double ty = cy - r.getCenterY();
        area.transform(AffineTransform.getTranslateInstance(tx,ty));
        
        return area;
    }
    
    @Override
    public void showOptionDialog(Component parentComponent) {
        HeartSymbolOptionPane option = new HeartSymbolOptionPane();
        option.setValue(radius);
        int choice = JOptionPane.showConfirmDialog(parentComponent,option,getTooltipText(),JOptionPane.OK_CANCEL_OPTION, JOptionPane.PLAIN_MESSAGE);
        if (choice == JOptionPane.OK_OPTION) {
            try {
                option.validateValue();
            } catch (Exception ex) {
                JOptionPane.showMessageDialog(parentComponent,ex.getMessage(),getTooltipText(), JOptionPane.ERROR_MESSAGE);
            }
            radius = option.getRadius();
        }
    }
}
